#!/usr/bin/env python3
"""
order_by_dynamic_range.py
Analyze .mp3 files, create a varied-flow order (alternating high/low dynamic range),
and rename files with zero-padded numeric prefixes: 001_, 002_, ...

Metric:
- Uses librosa to load audio.
- DR score = crest factor (peak vs. RMS, in dB) + RMS variability (std of frame RMS in dB).
  Not perfect LRA, but stable and fast for large batches.

Usage:
  python order_by_dynamic_range.py /path/to/mp3s
  python order_by_dynamic_range.py /path/to/mp3s --dry-run
  python order_by_dynamic_range.py /path/to/mp3s --prefix-width 3 --pattern "{num}_{name}"

Notes:
- Default rename pattern: "{num}_{name}" where num is zero-padded (e.g., 001_MySong.mp3)
- Safe: won’t overwrite; writes a CSV log: dynamic_order_log.csv
"""

import argparse
import math
import sys
from pathlib import Path
import re
import numpy as np
import pandas as pd
import librosa

def analyze_file(path: Path, hop_length=1024):
    """Return dict with DR metrics for an audio file."""
    try:
        # Load mono at native sr for speed (sr=None), let librosa resample? Better: keep native.
        y, sr = librosa.load(path.as_posix(), sr=None, mono=True)
        if y.size == 0:
            raise ValueError("Empty audio")
        # Peak and RMS
        peak = float(np.max(np.abs(y)) + 1e-12)
        rms = float(np.sqrt(np.mean(y**2)) + 1e-12)
        crest_db = 20.0 * math.log10(peak / rms)  # higher => more dynamic headroom

        # Frame-wise RMS to gauge variability (how much music breathes)
        frame_rms = librosa.feature.rms(y=y, frame_length=2048, hop_length=hop_length, center=True)[0]
        # Convert to dBFS-ish; avoid log(0)
        frame_rms_db = 20.0 * np.log10(np.maximum(frame_rms, 1e-12))
        rms_var_db = float(np.std(frame_rms_db))

        # Composite DR score: weight both aspects
        dr_score = crest_db + 0.6 * rms_var_db

        dur = len(y) / sr
        return {
            "file": path.name,
            "peak": peak,
            "rms": rms,
            "crest_db": crest_db,
            "rms_var_db": rms_var_db,
            "dr_score": dr_score,
            "duration_s": dur
        }
    except Exception as e:
        return {"file": path.name, "error": str(e)}

def alternating_extremes_order(rows):
    """
    Given rows sorted by dr_score descending, interleave high/low:
    [H1, H2, H3, ...], [L1, L2, L3, ...] -> H1, L1, H2, L2, ...
    This maximizes contrast between neighbors.
    """
    rows_sorted = sorted(rows, key=lambda r: r["dr_score"], reverse=True)
    hi = rows_sorted[:len(rows_sorted)//2 + len(rows_sorted)%2]
    lo = list(reversed(rows_sorted[len(rows_sorted)//2 + len(rows_sorted)%2:]))

    ordered = []
    for i in range(max(len(hi), len(lo))):
        if i < len(hi):
            ordered.append(hi[i])
        if i < len(lo):
            ordered.append(lo[i])
    return ordered

def strip_existing_prefix(name: str):
    """
    Remove existing numeric prefix like:
      '001_', '01 - ', '12. ', '003 ' etc.
    Returns (base_without_prefix, had_prefix_bool)
    """
    # Patterns: start digits + separators
    m = re.match(r'^(\d{1,4})[\s._-]+(.*)$', name)
    if m:
        return m.group(2), True
    return name, False

def main():
    ap = argparse.ArgumentParser(description="Order MP3s by dynamic range and prefix numbers.")
    ap.add_argument("folder", help="Folder containing .mp3 files")
    ap.add_argument("--dry-run", action="store_true", help="Show plan only; do not rename files")
    ap.add_argument("--prefix-width", type=int, default=3, help="Zero-pad width (default: 3 → 001)")
    ap.add_argument("--pattern", type=str, default="{num}_{name}",
                    help="Rename pattern; placeholders: {num}, {name}. Default: '{num}_{name}'")
    ap.add_argument("--glob", type=str, default="*.mp3", help="Glob pattern (default: *.mp3)")
    args = ap.parse_args()

    folder = Path(args.folder).expanduser().resolve()
    if not folder.is_dir():
        print(f"Error: {folder} is not a directory", file=sys.stderr)
        sys.exit(1)

    files = sorted([p for p in folder.glob(args.glob) if p.is_file()])
    if not files:
        print("No MP3 files found.", file=sys.stderr)
        sys.exit(1)

    print(f"Analyzing {len(files)} file(s) … (this can take a minute)")

    rows = []
    for p in files:
        info = analyze_file(p)
        info["path"] = p.as_posix()
        rows.append(info)

    # Report errors early
    errs = [r for r in rows if "error" in r]
    if errs:
        print("\nSome files could not be analyzed:")
        for e in errs:
            print(f" - {e['file']}: {e['error']}")
        rows = [r for r in rows if "error" not in r]
        if not rows:
            sys.exit(1)

    # Build order
    ordered = alternating_extremes_order(rows)

    # Prepare rename plan
    total = len(ordered)
    plan = []
    for idx, r in enumerate(ordered, start=1):
        original_name = r["file"]
        base_no_ext = Path(original_name).stem
        ext = Path(original_name).suffix  # .mp3
        # Strip existing numeric prefix safely
        base_no_ext_stripped, _ = strip_existing_prefix(base_no_ext)
        num = str(idx).zfill(args.prefix_width)
        new_base = args.pattern.format(num=num, name=base_no_ext_stripped)
        # Ensure extension preserved
        new_name = f"{new_base}{ext}"
        src = folder / original_name
        dst = folder / new_name
        plan.append({
            "index": idx,
            "src": src,
            "dst": dst,
            "dr_score": r["dr_score"],
            "crest_db": r["crest_db"],
            "rms_var_db": r["rms_var_db"],
            "duration_s": r["duration_s"]
        })

    # Check collisions
    dst_names = [p["dst"].name for p in plan]
    if len(dst_names) != len(set(dst_names)):
        print("Error: rename would create duplicate filenames. Try a different --pattern.", file=sys.stderr)
        sys.exit(1)

    # Show top/bottom few as preview
    print("\nPreview (first 10 in final order):")
    for p in plan[:10]:
        print(f" {p['index']:>3}: {Path(p['dst']).name}  "
              f"(DR={p['dr_score']:.2f}, crest={p['crest_db']:.2f} dB, var={p['rms_var_db']:.2f} dB)")

    # Write log
    log_path = folder / "dynamic_order_log.csv"
    df = pd.DataFrame([{
        "index": p["index"],
        "src": p["src"].name,
        "dst": p["dst"].name,
        "dr_score": p["dr_score"],
        "crest_db": p["crest_db"],
        "rms_var_db": p["rms_var_db"],
        "duration_s": p["duration_s"],
    } for p in plan])
    df.to_csv(log_path, index=False)
    print(f"\nWrote metrics/order log → {log_path.name}")

    if args.dry_run:
        print("\nDry run only. No files were renamed. Use without --dry-run to apply.")
        return

    # Rename (two-phase to avoid conflicts): temp names → final names
    temp_suffix = ".tmp_ren"
    temp_paths = []
    try:
        # Phase 1: move each src to a guaranteed-unique temp name
        for p in plan:
            temp = p["src"].with_name(p["src"].name + temp_suffix)
            p["src"].rename(temp)
            temp_paths.append((temp, p["dst"]))
        # Phase 2: move temp to final name
        for temp, final in temp_paths:
            if final.exists():
                raise FileExistsError(f"Destination exists: {final}")
            temp.rename(final)
        print("\nRenaming complete ✅")
    except Exception as e:
        print(f"\nError during rename: {e}", file=sys.stderr)
        print("Attempting to roll back…", file=sys.stderr)
        # Rollback temps back to original names where possible
        for temp, final in temp_paths:
            try:
                if temp.exists():
                    # Find original src from final name by removing prefix if needed
                    # Safer: move back to folder with a 'RESTORE_' prefix
                    restore = temp.with_name(re.sub(rf'{re.escape(temp_suffix)}$', '', temp.name))
                    temp.rename(restore)
                # If temp already moved to final, leave it; user can fix manually.
            except Exception:
                pass
        sys.exit(1)

if __name__ == "__main__":
    main()
